#!/usr/bin/python

#
# Copyright 2008 Amit Shrestha
#

'''
This file is part of Words.

Words is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Words is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Words.  If not, see <http://www.gnu.org/licenses/>.
'''

from javax.swing import JFrame
from javax.swing import JLabel
from javax.swing import JPanel
from javax.swing import JScrollPane
from java.awt import Toolkit

from java.awt import RenderingHints
from java.awt import Color
from java.awt import Font
from java.awt import BasicStroke
from java.awt import Image

from java.awt.event import KeyListener
from java.awt.event import KeyEvent

from javax.swing.border import EmptyBorder

from java.awt import GridBagLayout
from java.awt import GridBagConstraints
from java.awt import Insets

import random

from gui.utility import *
from gui.HTMLBox import HTMLBox

from listeners.ComponentListenerHangman import ComponentListenerHangman

class Hangman(JFrame,KeyListener):
    def __init__(self,main_class):
        JFrame.__init__(self)
        
        self.main_class = main_class
        self.words = self.main_class.db.words
        self.indices = self.main_class.db.get_indices(
            self.main_class.active_index_name
        )
        
        self.__add_controls()
        self.misses = []
        self.hits = []
        self.html_box.addKeyListener(self)
        self.setTitle("Hangman")
        
        self.GAME_STATE_IN_PROGRESS = 0
        self.GAME_STATE_GAME_OVER   = 1
        self.GAME_STATE_GAVE_UP     = 2
        self.GAME_STATE_WON         = 3

        width = self.main_class.configurator.get_int("hangman_frame.width")
        height = self.main_class.configurator.get_int("hangman_frame.height")
        if width and height: self.setSize(width,height)
        else: self.setSize(500,400)

        self.addComponentListener(ComponentListenerHangman(self.main_class))
        
        self.setLocation(
            self.getToolkit().getScreenSize().width/2 - self.getWidth()/2,
            self.getToolkit().getScreenSize().height/2 - self.getHeight()/2
        )

        self.setVisible(True)
        self.start_game()   
        #set visible again ... "type a letter"
        #does not show up otherwise :( ?
        self.setVisible(True)
        
    def start_game(self):
        self.setup_new()

    def __del__(self):
        JFrame.__del__(self)

    def __add_controls(self):
        gbl = GridBagLayout()
        gbc = GridBagConstraints()
        self.setLayout(gbl)
       
        self.html_box = HTMLBox()
        build_gbc(gbc,0,0,1,1,100,30,GridBagConstraints.BOTH)
        scroll = JScrollPane(
            self.html_box,
            JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED,
            JScrollPane.HORIZONTAL_SCROLLBAR_NEVER
        )
        scroll.setBorder(EmptyBorder(0,0,0,0))
        self.getContentPane().add(scroll,gbc)
                
        self.drawing_box = JPanel()
        self.drawing_box.setBorder(EmptyBorder(0,0,0,0))
        self.drawing_box.setOpaque(True)
        self.drawing_box.setBackground(Color.white)
        build_gbc(gbc,0,1,1,1,100,60,GridBagConstraints.BOTH)
        scroll = JScrollPane(
            self.drawing_box,
            JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED,
            JScrollPane.HORIZONTAL_SCROLLBAR_NEVER
        )
        scroll.setBorder(EmptyBorder(0,0,0,0))
        self.getContentPane().add(scroll,gbc)
        
        lbl_command_help = JLabel("  Press ESC to reveal word, Enter for new word",JLabel.LEFT)
        lbl_command_help.setOpaque(True)
        lbl_command_help.setBackground(Color.white)
        build_gbc(gbc,0,2,1,1,100,10,GridBagConstraints.BOTH)
        self.getContentPane().add(lbl_command_help,gbc)
        
        icon = Toolkit.getDefaultToolkit().getImage("files/icons/hangman.png")
        self.setIconImage(icon)
    
    def update_drawing(self):
        width = self.drawing_box.getWidth()
        height = self.drawing_box.getHeight()
        
        offscreen_image = self.createImage(width,height) 
        g = offscreen_image.getGraphics()

        g.setBackground(Color.white)
        g.clearRect(0,0,width,height)

        rh = RenderingHints(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON)
        rh.put(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY)
        g.setRenderingHints(rh)

        bs_square_2 = BasicStroke(2,BasicStroke.CAP_SQUARE, BasicStroke.JOIN_BEVEL)
        bs_square_4 = BasicStroke(4,BasicStroke.CAP_SQUARE, BasicStroke.JOIN_BEVEL)
        bs_square_8 = BasicStroke(8,BasicStroke.CAP_SQUARE, BasicStroke.JOIN_BEVEL)
        bs_butt_8 = BasicStroke(8,BasicStroke.CAP_BUTT, BasicStroke.JOIN_BEVEL)

        g.setStroke(bs_square_8)
        d =[x for x in self.word_name if x not in self.hits] 
        if len(d) == 0:
            self.game_state = self.GAME_STATE_WON
            g.setColor(Color.red)
            g.setFont(Font("Purisa",Font.ITALIC,36))
            g.drawString("YOU WIN!",50,50)
            onscreen_g = self.drawing_box.getGraphics()
            onscreen_g.drawImage(offscreen_image,0,0,self)
            return
        elif self.game_state == self.GAME_STATE_GAVE_UP:
            g.setFont(Font("Purisa",Font.ITALIC,28))
            g.drawString("Word Revealed",50,50)
            onscreen_g = self.drawing_box.getGraphics()
            onscreen_g.drawImage(offscreen_image,0,0,self)
            return
        elif len(self.misses) < 9:
            g.setFont(Font("Purisa",Font.ITALIC,28))
            g.drawString("Type a letter",50,50)

        #pole
        if len(self.misses) >= 1:
            g.drawLine(width - 20, 5, width - 20, 100)
        #floor
        if len(self.misses) >= 2:
            g.drawLine(width,100,width-120,100)
        #roof
        if len(self.misses) >= 3:
            g.drawLine(width-20,5,width-80,5)
            #support
            g.setStroke(bs_butt_8)
            g.drawLine(width-30,5,width-20,15)
        #rope
        if len(self.misses) >= 4:
            g.setStroke(bs_square_4)
            g.drawLine(width-60,5,width-60,15)
        
        if len(self.misses) >= 9:
            g.setColor(Color.red)
            g.setFont(Font("Purisa",Font.ITALIC,36))
            g.drawString("Game Over",50,50)
            self.game_state = self.GAME_STATE_GAME_OVER
        
        #head
        if len(self.misses) >= 5:
            g.setStroke(bs_square_2)
            g.fillOval(width-70,15,20,20)
        #torso
        if len(self.misses) >= 6:
            g.drawLine(width-60,35,width-60,65)
        #hands
        if len(self.misses) >= 7:
            g.drawLine(width-75,50,width-45,50)
        #legs
        if len(self.misses) >= 8:
            g.drawLine(width-60,65,width-45,80)
        if len(self.misses) >= 9:
            g.drawLine(width-60,65,width-75,80)
        
        onscreen_g = self.drawing_box.getGraphics()
        onscreen_g.drawImage(offscreen_image,0,0,self)

    def paint(self,g):
        JFrame.paint(self,g)
        self.update_drawing()
    
    def setup_new(self):
        self.game_state = self.GAME_STATE_IN_PROGRESS
        self.misses = []
        self.hits = []

        rindex = random.randint(0,len(self.indices)-1)
        word = self.words[self.indices[rindex]]

        self.word_name = word.name.upper()
        mindex = random.randint(0,len(word.meanings)-1)
        self.word_hint = word.meanings[mindex][0]
        self.update_controls()
    
    def update_controls(self):
        problem = ""
        for x in self.word_name:
            if x in self.hits:
                problem += "%s " % x
            else:
                problem += "_ "

        if self.game_state != self.GAME_STATE_IN_PROGRESS:
            html = "<h3>Guess the word : %s</h3>" % self.word_name
            html += "<i>Hint</i> : %s" % self.word_hint
        else:
            html = "<h3>Guess the word : %s</h3>" % problem
            html += "<i>Hint</i> : %s" % self.word_hint
            html += "<h3>Misses : " + " ".join(self.misses) + "</h3>"
        self.html_box.setText(html)
        self.html_box.setCaretPosition(0)
        self.update_drawing()

    def keyPressed(self,e):
        key_code = e.getKeyCode()
        key_char = e.getKeyChar().upper()
        source = e.getSource()

        if key_code == KeyEvent.VK_ESCAPE:
            self.game_state = self.GAME_STATE_GAVE_UP
            self.update_controls()
        elif key_code == KeyEvent.VK_ENTER:
            self.setup_new()
        elif key_char.isalpha():
            if self.game_state != self.GAME_STATE_IN_PROGRESS:
                return
            if key_char in self.word_name:
                if key_char not in self.hits:
                    self.hits.append(key_char)
                    self.update_controls()
            elif key_char not in self.misses:
                self.misses.append(key_char)
                self.update_controls()
    
    def keyReleased(self,e):
        pass
    
    def keyTyped(self,e):
        pass
