#!/usr/bin/python

#
# Copyright 2008 Amit Shrestha
#

'''
This file is part of Words.

Words is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Words is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Words.  If not, see <http://www.gnu.org/licenses/>.
'''

import sys
import random
from os import path
from os import mkdir

from javax.swing import DefaultListModel
from javax.swing import JOptionPane

from core.DB import DB
from core.Configurator import Configurator

from gui.MainFrame import MainFrame
from gui.DictionaryFrame import DictionaryFrame
from games.Hangman import Hangman

from listeners.KeyListenerMain import KeyListenerMain
from listeners.KeyListenerDictionary import KeyListenerDictionary
from listeners.MenuListenerMain import MenuListenerMain
from listeners.MouseListenerMain import MouseListenerMain
from listeners.MouseListenerDictionary import MouseListenerDictionary
from listeners.MyHyperlinkListener import MyHyperlinkListener
from listeners.WindowListenerMain import WindowListenerMain
from listeners.ComponentListenerMain import ComponentListenerMain
from listeners.ComponentListenerDictionary import ComponentListenerDictionary

#
# settings
#
DB_FILEPATH_DEFAULT = 'files/db.txt'
INDEX_PATTERNS_FILEPATH_DEFAULT = 'files/patterns_to_index.txt'
CONFIG_FILEPATH_DEFAULT = 'files/config.txt'

USER_DIR = ".words"
DB_FILEPATH_USER = USER_DIR + '/db.txt'
INDEX_PATTERNS_FILEPATH_USER = USER_DIR + '/patterns_to_index.txt'
CONFIG_FILEPATH_USER = USER_DIR + '/config.txt'


class MainClass:
    def __init__(self,splasher):
        self.db = DB()
        self.configurator = Configurator()

        self.__read_files()
        
        self.number_of_choices = 5
        self.dframe = None
        
        index_names = self.db.get_available_index_names()
        index_names.sort()

        last_index_name = self.configurator.get_string("last_active_index_name")
        if last_index_name and (last_index_name in index_names) and len(self.db.get_indices(last_index_name)) > 0 :
            self.active_index_name = last_index_name
        else: self.active_index_name = index_names[0]
        self.main_frame = MainFrame(index_names,self.active_index_name)

        x = self.configurator.get_int("main_frame.x")
        y = self.configurator.get_int("main_frame.y")
        if x and y: self.main_frame.setLocation(x,y)

        width = self.configurator.get_int("main_frame.width")
        height = self.configurator.get_int("main_frame.height")
        if width and height: self.main_frame.setSize(width,height)
        else:self.main_frame.setSize(600,500)
        
        self.main_frame.show()
        
        self.__add_listeners()
        self.test()
        self.main_frame.panel_output.output_box.grabFocus()
        splasher.dispose()

    def __read_files(self):
        db_path = DB_FILEPATH_DEFAULT
        index_patterns_path = INDEX_PATTERNS_FILEPATH_DEFAULT
        config_path = CONFIG_FILEPATH_DEFAULT

        if path.exists(DB_FILEPATH_USER):
            print 'Loading previously saved DB...'
            db_path = DB_FILEPATH_USER
        elif path.exists(".db.txt"): #for backwards compatibility
            print 'Loading previously saved DB...'
            db_path = ".db.txt"
        else:
            print 'Loading default DB...'
        
        if path.exists(INDEX_PATTERNS_FILEPATH_USER):
            print 'Loading customized index patterns...'
            index_patterns_path = INDEX_PATTERNS_FILEPATH_USER
        elif path.exists(".patterns_to_index.txt"): #for backwards compatibility
            print 'Loading customized index patterns...'
            index_patterns_path = ".patterns_to_index.txt"
        else:
            print 'Loading default index patterns...'
        
        self.db.read_db(db_path,index_patterns_path)
        print 'Done loading DB and index patterns.'

        if path.exists(CONFIG_FILEPATH_USER):
            print 'Loading user configurations...'
            config_path = CONFIG_FILEPATH_USER
        else:
            print 'Loading default configurations...'
        
        self.configurator.read_configurations(config_path)
        print 'Done loading configurations.'


    def __get_next_test_word(self):
        indices = self.db.get_indices(self.active_index_name)
        if len(indices) <= 0 :
            html = '<h3>You have mastered all the words in current practice set!</h3><h3>Please choose a different practice set.</h3>'
            self.main_frame.panel_output.output_box.setText(html)
            return (-1,None)
        rindex = random.randint(0,len(indices)-1)
        return (indices[rindex],self.db.words[indices[rindex]])

    def __get_trick_words(self, n):
        test_words = []
        while len(test_words) != n:
            rindex = random.randint(0,len(self.db.words)-1)
            if test_words.count(self.db.words[rindex]) == 0:
                test_words.append(self.db.words[rindex])
        return test_words

    def __add_listeners(self):
        keyboard_listener = KeyListenerMain(self)
        hyperlink_listener = MyHyperlinkListener(self)
        mouse_listener = MouseListenerMain(self)
        window_listener = WindowListenerMain(self)
        component_listener = ComponentListenerMain(self)
        
        #need to be class level so that more items could be added dynamically
        self.menu_listener = MenuListenerMain(self,self.db.get_available_index_names())
        
        self.main_frame.addWindowListener(window_listener)
        self.main_frame.addComponentListener(component_listener)
        self.main_frame.panel_output.output_box.addKeyListener(keyboard_listener)
        self.main_frame.panel_output.output_box.addHyperlinkListener(hyperlink_listener)
        self.main_frame.panel_choices.choices_box.addKeyListener(keyboard_listener)
        self.main_frame.menu_bar.file_menu_practice_sets.addMouseListener(mouse_listener)
        for x in self.main_frame.menu_bar.menu_items:
            x.addActionListener(self.menu_listener)
    
    def get_word_html(self,word_index):
        word = self.db.words[word_index]
        output = '<b><u>%s</u></b>&nbsp;&nbsp;[ <a href="%s">Find definition in dict.org</a> ]<br/><br/>' % (word.name,word.name)
        if word.pronunciations:
            output += '<b>Pronunciation</b>&nbsp;&nbsp;&nbsp;%s <a href="pronunciation_guide">Help</a><br/>' % (",".join(word.pronunciations))
        output += '<b>Meanings</b>'
        output += '<br/><ol>'
        number_of_meanings = len(word.meanings)
        for i in range(number_of_meanings):
            number_of_examples = len(word.meanings[i][1])
            output += "<li>%s" % word.meanings[i][0]
            if(number_of_examples > 0): output += ". e.g. "
            for j in range(number_of_examples):
                output += "%s. " % word.meanings[i][1][j]
            output += "</li>"
        output += "</ol>"
        if word.synonyms:
            output += '<b>Synonyms</b><br/>&nbsp;&nbsp;%s<br/>' % (", ".join(['<a href="' + x + '">' + x + '</a>' for x in word.synonyms]))
        if word.antonyms:
            output += '<b>Antonyms</b><br/>&nbsp;&nbsp;%s<br/>' % (", ".join(['<a href="' + x + '">' + x + '</a>' for x in word.antonyms]))
        
        output += '<br/><b>Categories</b>&nbsp;&nbsp;[ <a href="add_category_%d">Add</a> ] :&nbsp;&nbsp;%s' % (
                   word_index,
                   ", ".join([x + ' (<a href="delete_category_' + x + '_' + str(word_index) + '">delete</a>)' for x in word.categories])
        )
        return output

    def get_statistics_html(self):
        number_of_words_tested = self.db.get_number_of_words_tested()
        number_of_words_mastered = self.db.get_number_of_words_mastered()
        
        if number_of_words_mastered > 4000: 
            grade = "A++"
        elif number_of_words_mastered > 3000:
            grade = "A!"
        elif number_of_words_mastered > 1500:
            grade = "B+!"
        elif number_of_words_mastered > 700:
            grade = "B!"
        elif number_of_words_mastered > 500:
            grade = "C"
        elif number_of_words_mastered > 300:
            grade = "D"
        else:
            grade = "Ungraded. You have to work more."

        html = "<html><body>"
        html += '<h3 bgcolor="rgb(200,200,200)">&nbsp;<b>Your Grade</b> : %s</h3>' % grade
        
        html += '<table width="100%">'
        html += "<tr><td>Number of words in database</td><td>%d</td></tr>" % len(self.db.words)
        html += "<tr><td>Number of words tested</td><td>%d</td></tr>" % number_of_words_tested
        html += "<tr><td>Number of words mastered</td><td>%d</td></tr>" % number_of_words_mastered 
        html += "<tr><td>Number of words yet to be mastered</td><td>%d</td></tr>" % (len(self.db.words) - number_of_words_mastered)
        html += "</table>"

        html += '<h3 bgcolor="rgb(200,200,200)">&nbsp;Words yet to be mastered by practice sets</h3>'
        html += '<table width="100%">'
        index_names = self.db.get_available_index_names()
        index_names.sort()
        for x in index_names:
            html += "<tr><td>%s</td><td>%d</td></tr>" % (x,len(self.db.get_indices(x)))
        html += "</table>"

        html += "</body></html>"

        return html

    def reveal_answer(self,feedback):
        if self.test_word == None: return
        output = '<h3>%s</h3>' % feedback
        output += self.get_word_html(self.test_word_index)
        self.main_frame.panel_output.output_box.setText(output)
        self.main_frame.panel_output.output_box.setCaretPosition(0)

    def update_labels(self):
        if self.test_word == None: return
        self.main_frame.panel_labels.lbl_test_word.setText(self.test_word.name)
        self.main_frame.panel_labels.lbl_practice_set.setText("Current Practice Set : " + self.active_index_name)
        self.main_frame.panel_labels.lbl_times_tested.setText("Number of times tested : " + self.test_word.stats['number_of_times_tested'])
        self.main_frame.panel_labels.lbl_words_mastered.setText(
            "%d of %d mastered (%.2f %%)" % (
                self.db.get_number_of_words_mastered(), 
                len(self.db.words), 
                self.db.get_number_of_words_mastered() / float(len(self.db.words)) * 100
            )
        )
    
    def open_dictionary(self):
        self.dframe = DictionaryFrame()
        width = self.configurator.get_int("dictionary_frame.width")
        height = self.configurator.get_int("dictionary_frame.height")
        
        if width and height: self.dframe.setSize(width,height)
        else: self.dframe.setSize(500,300)

        self.dframe.setLocation(
            self.dframe.getToolkit().getScreenSize().width/2 - self.dframe.getWidth()/2,
            self.dframe.getToolkit().getScreenSize().height/2 - self.dframe.getHeight()/2
        )
        self.dframe.setTitle("Dictionary")
        list_model = DefaultListModel()
        for word in self.db.words:
            list_model.addElement(word.name)
        self.dframe.list_box.setModel(list_model)
        
        keyboard_listener = KeyListenerDictionary(self)
        hyperlink_listener = MyHyperlinkListener(self)
        mouse_listener = MouseListenerDictionary(self)
        component_listener = ComponentListenerDictionary(self)

        self.dframe.result_box.addHyperlinkListener(hyperlink_listener)
        self.dframe.list_box.addMouseListener(mouse_listener)
        self.dframe.cmd_search.addMouseListener(mouse_listener)
        self.dframe.txt_search.addKeyListener(keyboard_listener)
        self.dframe.addComponentListener(component_listener)
        
        self.dframe.setVisible(True)
    
    def open_hangman(self):
        indices = self.db.get_indices(self.active_index_name)
        if len(indices) <= 0 :
            html = '<html><body><h3>You have mastered all the words in current practice set!</h3><h3>Please choose a different practice set.</h3></body></html>'
            JOptionPane.showMessageDialog(self.main_frame,html,"Unable to start Hangman",JOptionPane.INFORMATION_MESSAGE)
        else:
            hman = Hangman(self)
    
    def mark_mastered(self, feedback):
        if self.test_word == None: return
        self.db.mark_mastered(self.test_word_index)
        self.update_labels()
        self.reveal_answer(feedback)

    def delete_category_from_word(self,word_index,category):
        word = self.db.words[word_index]
        if self.db.delete_category_from_word(word_index,category):
            if word_index == self.test_word_index:
                self.reveal_answer("Category deleted.")
            return True
        else: return False


    def add_category_to_word(self,word_index,category):
        word = self.db.words[word_index]
        if self.db.add_category_to_word(word_index,category):
            radio_item = self.main_frame.menu_bar.add_practice_set(category)
            if radio_item: 
                radio_item.addActionListener(self.menu_listener)
                self.menu_listener.add_index_name(category)
            if word_index == self.test_word_index:
                self.reveal_answer("Category added.")
            return True
        else: return False

    def set_active_index_name(self, index_name):
        if self.active_index_name != index_name:
            self.active_index_name = index_name
            self.configurator.set("last_active_index_name",index_name)
            self.test()

    def test(self):
        help_html = "<h3>Make a guess by typing a choice number, or hit ESC to give up.</h3><h3>To move to the next word, press Enter.</h3>"
        help_html += "<p><b>m</b> to mark current word as mastered. <b>d</b> to open dictionary.</p>"
        self.main_frame.panel_output.output_box.setText(help_html)
        (self.test_word_index, self.test_word) = self.__get_next_test_word()
        if self.test_word == None : return False
        trick_words = self.__get_trick_words(self.number_of_choices -1)
        self.update_labels()
        self.correct_choice = random.randint(1,self.number_of_choices)

        self.test_word.stats['number_of_times_tested'] = str(int(self.test_word.stats['number_of_times_tested']) + 1)
        trick_word_index = 0;
        choices = "<table>"
        for x in range(self.number_of_choices):
            choices += "<tr>"
            choices += '<td valign="top" width="20px"><b>%d</b>.<td>' % (x+1)
            if x != self.correct_choice-1:
                mindex = random.randint(0,len(trick_words[trick_word_index].meanings)-1)
                choices += '<td valign="top">' + trick_words[trick_word_index].meanings[mindex][0] + "</td>"
                trick_word_index += 1
            else:
                mindex = random.randint(0,len(self.test_word.meanings)-1)
                choices += "<td>" + self.test_word.meanings[mindex][0] + "</td>"
            choices += "</tr>"
        choices += "</table>"
        self.main_frame.panel_choices.choices_box.setText(choices)
        self.main_frame.panel_choices.choices_box.setCaretPosition(0)
        return True

    def save(self):
        try:
            if not path.exists(USER_DIR):
                mkdir(USER_DIR)
            self.db.write_db(DB_FILEPATH_USER,INDEX_PATTERNS_FILEPATH_USER)
            self.configurator.save_configurations(CONFIG_FILEPATH_USER)
            return True
        except:
            return False
